---
name: r.grow.py
description: Generates a raster map layer with contiguous areas grown by one cell.
keywords: [ raster, distance, proximity ]
---

# r.grow.py

Generates a raster map layer with contiguous areas grown by one cell.

=== "Command line"

    **r.grow.py**
    [**-m**]
    **input**=*name*
    **output**=*name*
    [**radius**=*float*]
    [**metric**=*string*]
    [**old**=*integer*]
    [**new**=*integer*]
    [**nprocs**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.grow.py input=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.grow.py***",
        **input**,
        **output**,
        **radius**=*1.01*,
        **metric**=*"euclidean"*,
        **old**=*None*,
        **new**=*None*,
        **nprocs**=*0*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.grow.py", input="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_grow_py*(**input**,
        **output**,
        **radius**=*1.01*,
        **metric**=*"euclidean"*,
        **old**=*None*,
        **new**=*None*,
        **nprocs**=*0*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_grow_py(input="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **radius**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Radius of buffer in raster cells  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.01*  
    **metric**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Metric  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *euclidean, maximum, manhattan*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *euclidean*  
    **old**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value to write for input cells which are non-NULL (-1 =&gt; NULL)  
    **new**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value to write for "grown" cells  
    **nprocs**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;0: use OpenMP default; &gt;0: use nprocs; &lt;0: use MAX-nprocs  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **-m**  
    &nbsp;&nbsp;&nbsp;&nbsp;Radius is in map units rather than cells  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **radius** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Radius of buffer in raster cells  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.01*  
    **metric** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Metric  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *euclidean, maximum, manhattan*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *euclidean*  
    **old** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value to write for input cells which are non-NULL (-1 =&gt; NULL)  
    **new** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value to write for "grown" cells  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;0: use OpenMP default; &gt;0: use nprocs; &lt;0: use MAX-nprocs  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Radius is in map units rather than cells  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **radius** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Radius of buffer in raster cells  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.01*  
    **metric** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Metric  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *euclidean, maximum, manhattan*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *euclidean*  
    **old** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value to write for input cells which are non-NULL (-1 =&gt; NULL)  
    **new** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value to write for "grown" cells  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;0: use OpenMP default; &gt;0: use nprocs; &lt;0: use MAX-nprocs  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Radius is in map units rather than cells  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.grow* adds cells around the perimeters of all areas in a
user-specified raster map layer and stores the output in a new raster
map layer. The user can use it to grow by one or more than one cell (by
varying the size of the **radius** parameter), or like *r.buffer*, but
with the option of preserving the original cells (similar to combining
*r.buffer* and *r.patch*).

If **radius** is negative,*r.grow* shrinks areas by removing cells
around the perimeters of all areas.

## NOTES

The user has the option of specifying three different metrics which
control the geometry in which grown cells are created, (controlled by
the **metric** parameter): *Euclidean*, *Manhattan*, and *Maximum*.

The *Euclidean distance* or *Euclidean metric* is the "ordinary"
distance between two points that one would measure with a ruler, which
can be proven by repeated application of the Pythagorean theorem. The
formula is given by:

```sh
d(dx,dy) = sqrt(dx^2 + dy^2)
```

Cells grown using this metric would form isolines of distance that are
circular from a given point, with the distance given by the **radius**.

The *Manhattan metric*, or *Taxicab geometry*, is a form of geometry in
which the usual metric of Euclidean geometry is replaced by a new metric
in which the distance between two points is the sum of the (absolute)
differences of their coordinates. The name alludes to the grid layout of
most streets on the island of Manhattan, which causes the shortest path
a car could take between two points in the city to have length equal to
the points' distance in taxicab geometry. The formula is given by:

```sh
d(dx,dy) = abs(dx) + abs(dy)
```

where cells grown using this metric would form isolines of distance that
are rhombus-shaped from a given point.

The *Maximum metric* is given by the formula

```sh
d(dx,dy) = max(abs(dx),abs(dy))
```

where the isolines of distance from a point are squares.

If there are two cells which are equal candidates to grow into an empty
space, *r.grow* will choose the northernmost candidate; if there are
multiple candidates with the same northing, the westernmost is chosen.

## EXAMPLE

In this example, the lakes map in the North Carolina sample dataset is
buffered:

```sh
g.region raster=lakes -p
# the lake raster map pixel resolution is 10m
r.grow input=lakes output=lakes_grown_100m radius=10
```

Shrinking instead of growing:

```sh
g.region raster=lakes -p
# the lake raster map pixel resolution is 10m
r.grow input=lakes output=lakes_shrunk_100m radius=-10
```

## SEE ALSO

*[r.buffer](r.buffer.md), [r.grow.distance](r.grow.distance.md),
[r.patch](r.patch.md)*

*[Wikipedia Entry: Euclidean
Metric](https://en.wikipedia.org/wiki/Euclidean_metric)*  
*[Wikipedia Entry: Manhattan
Metric](https://en.wikipedia.org/wiki/Manhattan_metric)*

## AUTHORS

Marjorie Larson, U.S. Army Construction Engineering Research Laboratory

Glynn Clements

## SOURCE CODE

Available at: [r.grow source code](https://github.com/OSGeo/grass/tree/main/scripts/r.grow)
([history](https://github.com/OSGeo/grass/commits/main/scripts/r.grow))  
Latest change: Thursday Nov 06 15:49:48 2025 in commit [1bcf99a](https://github.com/OSGeo/grass/commit/1bcf99adefa2c564346fa56847d783186cb9fcda)
