---
name: r.regression.multi
description: Calculates multiple linear regression from raster maps.
keywords: [ raster, statistics, regression ]
---

# r.regression.multi

Calculates multiple linear regression from raster maps.

=== "Command line"

    **r.regression.multi**
    [**-g**]
    **mapx**=*name* [,*name*,...]
    **mapy**=*name*
    [**residuals**=*name*]
    [**estimates**=*name*]
    [**output**=*name*]
    **format**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.regression.multi mapx=name mapy=name format=shell
    ```

=== "Python (grass.script)"

    *grass.script.parse_command*("***r.regression.multi***",
        **mapx**,
        **mapy**,
        **residuals**=*None*,
        **estimates**=*None*,
        **output**=*None*,
        **format**=*"shell"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.parse_command("r.regression.multi", mapx="name", mapy="name", format="json")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_regression_multi*(**mapx**,
        **mapy**,
        **residuals**=*None*,
        **estimates**=*None*,
        **output**=*None*,
        **format**=*"shell"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_regression_multi(mapx="name", mapy="name", format="json")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **mapx**=*name* [,*name*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for x coefficient  
    **mapy**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for y coefficient  
    **residuals**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store residuals  
    **estimates**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store estimates  
    **output**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;ASCII file for storing regression coefficients (output to screen if file not specified).  
    **format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *shell, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *shell*  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    **-g**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print in shell script style [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=shell instead.  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **mapx** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for x coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mapy** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for y coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **residuals** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store residuals  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **estimates** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store estimates  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;ASCII file for storing regression coefficients (output to screen if file not specified).  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *shell, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *shell*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print in shell script style [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=shell instead.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **mapx** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for x coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mapy** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map for y coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **residuals** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store residuals  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **estimates** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store estimates  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;ASCII file for storing regression coefficients (output to screen if file not specified).  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *shell, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *shell*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print in shell script style [deprecated]  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;This flag is deprecated and will be removed in a future release. Use format=shell instead.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.regression.multi* calculates a multiple linear regression from raster
maps, according to the formula

```sh
Y = b0 + sum(bi*Xi) + E
```

where

```sh
X = {X1, X2, ..., Xm}
m = number of explaining variables
Y = {y1, y2, ..., yn}
Xi = {xi1, xi2, ..., xin}
E = {e1, e2, ..., en}
n = number of observations (cases)
```

In R notation:

```sh
Y ~ sum(bi*Xi)
b0 is the intercept, X0 is set to 1
```

*r.regression.multi* is designed for large datasets that can not be
processed in R. A p value is therefore not provided, because even very
small, meaningless effects will become significant with a large number
of cells. Instead it is recommended to judge by the estimator b, the
amount of variance explained (R squared for a given variable) and the
gain in AIC (AIC without a given variable minus AIC global must be
positive) whether the inclusion of a given explaining variable in the
model is justified.

### The global model

The *b* coefficients (b0 is offset), R squared or coefficient of
determination (Rsq) and F are identical to the ones obtained from
R-stats's lm() function and R-stats's anova() function. The AIC value is
identical to the one obtained from R-stats's stepAIC() function (in case
of backwards stepping, identical to the Start value). The AIC value
corrected for the number of explaining variables and the BIC (Bayesian
Information Criterion) value follow the logic of AIC.

### The explaining variables

R squared for each explaining variable represents the additional amount
of explained variance when including this variable compared to when
excluding this variable, that is, this amount of variance is explained
by the current explaining variable after taking into consideration all
the other explaining variables.

The F score for each explaining variable allows testing if the inclusion
of this variable significantly increases the explaining power of the
model, relative to the global model excluding this explaining variable.
That means that the F value for a given explaining variable is only
identical to the F value of the R-function *summary.aov* if the given
explaining variable is the last variable in the R-formula. While R
successively includes one variable after another in the order specified
by the formula and at each step calculates the F value expressing the
gain by including the current variable in addition to the previous
variables, *r.regression.multi* calculates the F-value expressing the
gain by including the current variable in addition to all other
variables, not only the previous variables.

The AIC value is identical to the one obtained from the R-function
stepAIC() when excluding this variable from the full model. The AIC
value corrected for the number of explaining variables and the BIC value
(Bayesian Information Criterion) value follow the logic of AIC. BIC is
identical to the R-function stepAIC with k = log(n). AICc is not
available through the R-function stepAIC.

## EXAMPLE

Multiple regression with soil K-factor and elevation, aspect, and slope
(North Carolina dataset). Output maps are the residuals and estimates:

```sh
g.region raster=soils_Kfactor -p
r.regression.multi mapx=elevation,aspect,slope mapy=soils_Kfactor \
  residuals=soils_Kfactor.resid estimates=soils_Kfactor.estim
```

Using the JSON format option and Python to parse the output:

```python
import grass.script as gs

data = gs.parse_command(
    "r.regression.multi",
    mapx=["elevation", "aspect", "slope"],
    mapy="soils_Kfactor",
    format="json",
)
print(data)
```

Possible JSON Output:

```json
{
 "n": 525000,
 "Rsq": 0.115424,
 "Rsqadj": 0.115419,
 "RMSE": 0.026785,
 "MAE": 0.020049,
 "F": 22834.749577,
 "b0": 0.050201,
 "AIC": -3800909.191798,
 "AICc": -3800909.191798,
 "BIC": -3800864.507184,
 "predictors": [
  {
   "name": "elevation",
   "b": 0.001523,
   "F": 64152.655957,
   "AIC": -3740385.046757,
   "AICc": -3740385.046757,
   "BIC": -3740364.70445
  },
  {
   "name": "aspect",
   "b": 2.2e-05,
   "F": 4922.480908,
   "AIC": -3796011.607461,
   "AICc": -3796011.607461,
   "BIC": -3795991.265154
  },
  {
   "name": "slope",
   "b": 0.002853,
   "F": 11927.479106,
   "AIC": -3789117.096287,
   "AICc": -3789117.096287,
   "BIC": -3789096.75398
  }
 ]
}
```

## SEE ALSO

*[d.correlate](d.correlate.md),
[r.regression.line](r.regression.line.md), [r.stats](r.stats.md)*

## AUTHOR

Markus Metz

## SOURCE CODE

Available at: [r.regression.multi source code](https://github.com/OSGeo/grass/tree/main/raster/r.regression.multi)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.regression.multi))  
Latest change: Saturday Oct 11 04:38:40 2025 in commit [3dd71f5](https://github.com/OSGeo/grass/commit/3dd71f5fab83b72a2db2da959fe075c5b3b52200)
