---
name: v.net.steiner
description: Creates Steiner tree for the network and given terminals. Note that 'Minimum Steiner Tree' problem is NP-hard and heuristic algorithm is used in this module so the result may be sub optimal.
keywords: [ vector, network, steiner tree ]
---

# v.net.steiner

Creates Steiner tree for the network and given terminals.

Note that 'Minimum Steiner Tree' problem is NP-hard and heuristic algorithm is used in this module so the result may be sub optimal.

=== "Command line"

    **v.net.steiner**
    [**-g**]
    **input**=*name*
    **output**=*name*
    [**arc_type**=*string* [,*string*,...]]
    [**arc_layer**=*string*]
    [**node_layer**=*string*]
    [**acolumn**=*string*]
    **terminal_cats**=*range*
    [**npoints**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.net.steiner input=name output=name terminal_cats=range
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***v.net.steiner***",
        **input**,
        **output**,
        **arc_type**=*"line,boundary"*,
        **arc_layer**=*"1"*,
        **node_layer**=*"2"*,
        **acolumn**=*None*,
        **terminal_cats**,
        **npoints**=*-1*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("v.net.steiner", input="name", output="name", terminal_cats="range")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_net_steiner*(**input**,
        **output**,
        **arc_type**=*"line,boundary"*,
        **arc_layer**=*"1"*,
        **node_layer**=*"2"*,
        **acolumn**=*None*,
        **terminal_cats**,
        **npoints**=*-1*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_net_steiner(input="name", output="name", terminal_cats="range")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map  
    **arc_type**=*string* [,*string*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc type  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *line, boundary*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *line,boundary*  
    **arc_layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **node_layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Node layer (used for terminals)  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **acolumn**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arcs' cost column (for both directions)  
    **terminal_cats**=*range* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Category values  
    &nbsp;&nbsp;&nbsp;&nbsp;Categories of points on terminals (layer is specified by nlayer)  
    **npoints**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of Steiner points (-1 for all possible)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-1*  
    **-g**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use geodesic calculation for longitude-latitude projects  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **arc_type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc type  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *line, boundary*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *line,boundary*  
    **arc_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **node_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Node layer (used for terminals)  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **acolumn** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arcs' cost column (for both directions)  
    **terminal_cats** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Category values  
    &nbsp;&nbsp;&nbsp;&nbsp;Categories of points on terminals (layer is specified by nlayer)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, cats, *range*  
    **npoints** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of Steiner points (-1 for all possible)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use geodesic calculation for longitude-latitude projects  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **arc_type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc type  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *line, boundary*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *line,boundary*  
    **arc_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **node_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Node layer (used for terminals)  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **acolumn** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arcs' cost column (for both directions)  
    **terminal_cats** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Category values  
    &nbsp;&nbsp;&nbsp;&nbsp;Categories of points on terminals (layer is specified by nlayer)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, cats, *range*  
    **npoints** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of Steiner points (-1 for all possible)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use geodesic calculation for longitude-latitude projects  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.net.steiner* calculates the optimal connection of nodes on a vector
network.

A Steiner tree is used to calculate the minimum-cost vector network
connecting some number of end nodes in a network framework. For example
it could be used to find the path following a road system which will
minimize the amount of fibre optic cable needed to connect a series of
satellite offices.

Costs may be either line lengths, or attributes saved in a database
table. These attribute values are taken as costs of whole segments, not
as costs to traverse a length unit (e.g. meter) of the segment. For
example, if the speed limit is 100 km / h, the cost to traverse a 10 km
long road segment must be calculated as length / speed = 10 km / (100
km/h) = 0.1 h. Supported are cost assignments for both arcs and nodes.
For areas, costs will be calculated along boundary lines.

Points representing nodes must be exactly on network nodes, and the
input vector map needs to be prepared with *v.net operation=connect*.

## NOTES

Current implementation of obtaining Steiner tree is not memory
efficient. An attempt to run module on a network with large number of
intersections thus might result in failure to allocate memory or out of
memory condition.

## EXAMPLE

Steiner tree for 6 digitized nodes (Spearfish):

Shortest path, along unimproved roads:

![v.net.steiner example using distance](vnetsteiner.png)

Fastest path, along highways:

![v.net.steiner example using time](vnetsteinertime.png)

```sh
# Spearfish

g.copy vect=roads,myroads

# we have 6 locations to allocate
echo "1|601653.5|4922869.2|a
2|608284|4923776.6|b
3|601845|4914981.9|c
4|596270|4917456.3|d
5|593330.8|4924096.6|e
6|598005.5|4921439.2|f" | v.in.ascii in=- cat=1 x=2 y=3 out=centers col="cat integer, \
                         east double precision, north double precision, label varchar(43)"

v.db.select centers
v.category centers op=report
# type       count        min        max
# point          6          1          6

# create lines map connecting points to network (on layer 2)
v.net myroads points=centers out=myroads_net op=connect thresh=500

# set up costs as traveling time

# create unique categories for each road in layer 3
v.category in=myroads_net out=myroads_net_time opt=add cat=1 layer=3 type=line

# add new table for layer 3
v.db.addtable myroads_net_time layer=3 col="cat integer,label varchar(43),length double precision,speed double precision,cost double precision"

# copy road type to layer 3
v.to.db myroads_net_time layer=3 qlayer=1 opt=query qcolumn=label columns=label

# upload road length in miles
v.to.db myroads_net_time layer=3 type=line option=length col=length unit=miles

# set speed limits in miles / hour
v.db.update myroads_net_time layer=3 col=speed val="5.0"
v.db.update myroads_net_time layer=3 col=speed val="75.0" where="label='interstate'"
v.db.update myroads_net_time layer=3 col=speed val="75.0" where="label='primary highway, hard surface'"
v.db.update myroads_net_time layer=3 col=speed val="50.0" where="label='secondary highway, hard surface'"
v.db.update myroads_net_time layer=3 col=speed val="25.0" where="label='light-duty road, improved surface'"
v.db.update myroads_net_time layer=3 col=speed val="5.0" where="label='unimproved road'"

# define traveling costs as traveling time in minutes:
v.db.update myroads_net_time layer=3 col=cost val="length / speed * 60"

# shortest path
v.net.steiner myroads_net_time arc_layer=3 node_layer=2 terminal_cats=1-6 out=mysteiner_distance

# fastest path
v.net.steiner myroads_net_time arc_layer=3 node_layer=2 acol=cost terminal_cats=1-6 out=mysteiner_time
```

To display the result, run for example:

```sh
# display the results
g.region vector=myroads_net

# shortest path
d.mon x0
d.vect myroads_net
d.vect -c centers icon=basic/triangle
d.font Vera
d.vect centers col=red disp=attr attrcol=label lsize=12
d.vect mysteiner_distance col=blue width=2

# fastest path
d.mon x1
d.vect myroads_net
d.vect -c centers icon=basic/triangle
d.font Vera
d.vect centers col=red disp=attr attrcol=label lsize=12
d.vect mysteiner_time col=blue width=2
```

## SEE ALSO

*[d.path](d.path.md), [v.net](v.net.md), [v.net.alloc](v.net.alloc.md),
[v.net.iso](v.net.iso.md), [v.net.path](v.net.path.md),
[v.net.salesman](v.net.salesman.md)*

## AUTHORS

Radim Blazek, ITC-Irst, Trento, Italy  
Documentation: Markus Neteler, Markus Metz

## SOURCE CODE

Available at: [v.net.steiner source code](https://github.com/OSGeo/grass/tree/main/vector/v.net.steiner)
([history](https://github.com/OSGeo/grass/commits/main/vector/v.net.steiner))  
Latest change: Thursday Feb 13 14:10:24 2025 in commit [c0fe7d7](https://github.com/OSGeo/grass/commit/c0fe7d746d0e4b334ec160652469b3ced51e03b2)
